//
//	Code to setup and draw display
//
//        Copyright (c) 2002-2005 Jim Peters <http://uazu.net/>.
//        Released under the GNU GPL version 2 as published by the
//        Free Software Foundation.  See the file COPYING for details,
//        or visit <http://www.gnu.org/copyleft/gpl.html>.
//

#include "all.h"

//
//	Arrange the display, based on current display size and font
//	size
//

void 
arrange_display() {
   int big= disp_font == 16;

   d_tim_xx= 0;
   d_tim_yy= 0;
   d_tim_sx= 6 * (big ? 8 : 6);
   d_tim_sy= disp_font;

   d_vsc_xx= 0;
   d_vsc_yy= d_tim_yy + d_tim_sy;
   d_vsc_sx= d_tim_sx;
   d_vsc_sy= disp_sy - disp_font - d_vsc_yy;

   d_hsc_xx= d_tim_xx + d_tim_sx;
   d_hsc_yy= d_tim_yy;
   d_hsc_sx= disp_sx - d_hsc_xx;
   d_hsc_sy= d_tim_sy;

   d_mag_xx= d_hsc_xx;
   d_mag_yy= d_vsc_yy;
   d_mag_sx= d_hsc_sx;
   d_mag_sy= d_vsc_sy;
}

//
//	Draw status line (redraw it, really)
//

static char *status_str= 0;

void 
draw_status() {
   drawtext(disp_font, 0, disp_sy-disp_font, status_str);
   update(0, disp_sy-disp_font, disp_sx, disp_font);
}

//
//      Display a status line.  Colours may be selected using
//      characters from 128 onwards (see drawtext()).  There are two
//      types of status lines -- temporary ones and permanent ones.
//      Permanent ones have a '+' at the front of the formatted text
//      (although this is not displayed).  If the status is cleared
//      using status(""), then a permanent message will not go away.
//      However, it will go away if any other message is displayed,
//      including status("+").
//

void
status(char *fmt, ...) {
   va_list ap;
   char buf[4096], *p;
   static int c_perm= 0;
   int perm= 0;

   if (fmt[0] == '+') { fmt++; perm= 1; }
   if (!fmt[0] && c_perm && !perm) return;
   c_perm= perm;

   va_start(ap, fmt);
   buf[0]= 128;         // Select white on black
   vsprintf(buf+1, fmt, ap);
   p= strchr(buf, 0);
   *p++= 0x80;          // Restore white on black
   *p++= '\n';          // Blank to end of line
   *p= 0;

   if (status_str) free(status_str);
   status_str= StrDup(buf);
   draw_status();
}


//
//	Draw vertical scale based on current file 'curr'
//

void 
draw_vsc() {
   double off0= curr->midy - curr->scale * YFACT * 0.5 * d_vsc_sy;
   double off1= curr->midy + curr->scale * YFACT * 0.5 * d_vsc_sy;
   double step= (off1 - off0)/5;
   int log10= (int)floor(log(step) / M_LN10);
   double step10= exp(M_LN10 * log10);
   double off;
   int fwid= disp_font == 16 ? 8 : 6;
   
   if (step >= step10 * 5) step= step10*5;
   else if (step >= step10 * 2) step= step10*2;
   else step= step10;

   off= step * floor((off0 + step * 0.999) / step);

   clear_rect(d_vsc_xx, d_vsc_yy, d_vsc_sx, d_vsc_sy, colour[4]);

   while (off < off1) {
      int yy= d_vsc_yy + (int)((off-off0) / (off1-off0) * d_vsc_sy);
      char buf[32];
      
      hline(d_vsc_xx + d_vsc_sx - fwid, yy, fwid, colour[9]);

      if (log10 < -4) 
	 sprintf(buf, "\x9A%g", -off);
      else if (log10 < 0)
	 sprintf(buf, "\x9A%.*f", -log10, -off);
      else 
	 sprintf(buf, "\x9A%.0f", -off);

      if (strlen(buf) > 6)
	 sprintf(buf, "\x9A??????");

      if (yy-disp_font > d_vsc_yy)
	 drawtext(disp_font, d_vsc_xx + d_vsc_sx - fwid * (strlen(buf)-1),
		  yy-disp_font, buf);
      off += step;
   }

   // Update
   update(d_vsc_xx, d_vsc_yy, d_vsc_sx, d_vsc_sy);
}


//
//	Draw horizontal scale based on current file 'curr'
//

void 
draw_hsc() {
   double off0= curr->midx - curr->scale * d_hsc_sx * 0.5;
   double off1= curr->midx + curr->scale * d_hsc_sx * 0.5;
   double step= (off1 - off0)/5;
   int log10= (int)floor(log(step) / M_LN10);
   double step10= exp(M_LN10 * log10);
   double off;
   
   if (step >= step10 * 5) step= step10*5;
   else if (step >= step10 * 2) step= step10*2;
   else step= step10;

   off= step * floor((off0 + step * 0.999) / step);

   clear_rect(d_hsc_xx, d_hsc_yy, d_hsc_sx, d_hsc_sy, colour[3]);

   while (off < off1) {
      int xx= d_hsc_xx + (int)((off-off0) / (off1-off0) * d_hsc_sx);
      char buf[32];
      
      vline(xx, d_hsc_yy + d_hsc_sy/2, d_hsc_sy/2, colour[9]);

      if (log10 < -4) 
	 sprintf(buf, "\x9A%g", off);
      else if (log10 < 0)
	 sprintf(buf, "\x9A%.*f", -log10, off);
      else 
	 sprintf(buf, "\x9A%.0f", off);

      drawtext(disp_font, xx + 3, d_hsc_yy, buf);
      off += step;
   }

   // Update
   update(d_hsc_xx, d_hsc_yy, d_hsc_sx, d_hsc_sy);
}


//
//	Draw time area, based on current file 'curr'
//

void 
draw_tim() {
   char buf[32];
   int tt= floor(curr->off * curr->h_tstep + curr->h_win*0.5);
   
   sprintf(buf, "\x92%2d:%02d ", tt/60, tt%60);
   buf[7]= 0;
   
   clear_rect(d_tim_xx, d_tim_yy, d_tim_sx, d_tim_sy, colour[0]);
   drawtext(disp_font, d_tim_xx, d_tim_yy, buf);

   // Update
   update(d_tim_xx, d_tim_yy, d_tim_sx, d_tim_sy);
}


//
//	Draw the magnitude area
//

void 
draw_mag() {
   int xx, yy;
   int unit= d_mag_sx/10;

   // Load data if necessary
   if (dat_file != curr || dat_off != curr->off)
      reload_dat();

   // Redo analysis if necessary
   do_analysis();
   
   switch (d_mode) {
    case -2:
       for (yy= 0; yy < d_mag_sy; yy++) 
	  for (xx= 0; xx<d_mag_sx; xx++) 
	     plot_cint_bar(d_mag_xx+xx, d_mag_yy+yy, 
			   -xx, unit, 
			   curr->bri * analysis[xx+yy*an_sx]);
       break;
    case -1:
       for (yy= 0; yy < d_mag_sy; yy++) 
	  for (xx= 0; xx<d_mag_sx; xx++) 
	     plot_cint_abar(d_mag_xx+xx, d_mag_yy+yy, 
			    -xx, -yy, unit, 
			    curr->bri * analysis[xx+yy*an_sx]);
       break;
    case 0:
       for (yy= 0; yy < d_mag_sy; yy++) 
	  for (xx= d_mag_sx-1; xx >= 0; xx--) 
	     plot_cint(d_mag_xx+xx, d_mag_yy+yy, 
		       curr->bri * analysis[xx+yy*an_sx]);
       break;
    case 1:
       for (yy= 0; yy < d_mag_sy; yy++) 
	  for (xx= d_mag_sx-1; xx >= 0; xx--) 
	     plot_cint_abar(d_mag_xx+xx, d_mag_yy+yy, 
			    d_mag_sx-xx, -yy, unit, 
			    curr->bri * analysis[xx+yy*an_sx]);
       break;
    case 2:
       for (yy= 0; yy < d_mag_sy; yy++) 
	  for (xx= d_mag_sx-1; xx >= 0; xx--) 
	     plot_cint_bar(d_mag_xx+xx, d_mag_yy+yy, 
			   d_mag_sx-xx, unit, 
			   curr->bri * analysis[xx+yy*an_sx]);
       break;
   }

   // Update
   update(d_mag_xx, d_mag_yy, d_mag_sx, d_mag_sy);
}

//
//	Reload raw data from the file
//

void 
reload_dat() {
   int magic;

   if (dat_file != curr && dat_in) {
      fclose(dat_in);
      dat_in= 0;
   }
   if (dat) {
      free(dat); 
      dat= 0;
   }

   //   warn("Reload offset %d of %s", curr->off, curr->fnam);

   dat_file= curr;
   dat_off= curr->off;
   if (!(dat= malloc(curr->h_cnt * 8)))
      error("Out of mem");

   if (!dat_in && !(dat_in= fopen(curr->path, "rb")))
      error("Problems reading file: %s", curr->fnam);

   // If off the end, just silently return zeros
   if (0 != fseek(dat_in, 48 + curr->off * (4 + curr->h_cnt * 8), SEEK_SET) ||
       1 != fread(&magic, 4, 1, dat_in)) {
      status("\x8A EOF \x80", curr->fnam);
      memset(dat, 0, curr->h_cnt * 8);
      return;
   } 

   // If bad magic or wrong length, warn of corruption
   if (magic != 0x12345678 ||
       1 != fread(dat, curr->h_cnt * 8, 1, dat_in)) {
      status("\x8AWARNING:\x80 Corrupt .bav file: \x8C%s\x80", curr->fnam);
      memset(dat, 0, curr->h_cnt * 8);
      return;
   }
}

// END //

