//
//	bavsa-view -- display the analysis from the bavsa binaural
//	  beat visual analysis tool
//
//        Copyright (c) 2002-2005 Jim Peters <http://uazu.net/>.
//        Released under the GNU GPL version 2 as published by the
//        Free Software Foundation.  See the file COPYING for details,
//        or visit <http://www.gnu.org/copyleft/gpl.html>.
//

// "Keys: Arrows: scroll display; Z,X: zoom,expand; A: auto-adjust display;"
// "Space,BackSp: forward,back; PgDn,PgUp: forward,back; [,]: prev,next file;"
// "0-9 select brightness; +,- increase,decrease brightness."

#ifdef HEADER

typedef struct {
   char h_magic[16];	// header starts here ...
   double h_step;	// frequency step
   int h_f0, h_cnt;	// start frequency, and count
   double h_win;	// window width
   double h_tstep;	// time step
   int h_end;		// header end

   char *fnam;		// filename
   char *path;		// full path to file
   double scale;	// Hz/pixel for horizontal scale; vertical is 0.1*scale Hz/pix
   double midx;		// mid-display carrier frequency
   double midy;		// mid-display beat frequency
   double bri;		// display brightness
   int off;		// current offset into file, in chunks
   int font;		// current font: 0 small, 1 big
   int file;		// current file number
} Settings;

// Factor that vertical (beat) scale is different to horizontal
#define YFACT 0.2

#else

#include "all.h"

//
//	Globals
//

// Display-related
SDL_Surface *disp;	// Display
Uint32 *disp_pix32;	// Pixel data for 32-bit display, else 0
Uint16 *disp_pix16;	// Pixel data for 16-bit display, else 0
int disp_my;            // Display pitch, in pixels
int disp_sx, disp_sy;   // Display size

int disp_rl, disp_rs;	// Red component transformation
int disp_gl, disp_gs;	// Green component transformation
int disp_bl, disp_bs;	// Blue component transformation
int disp_am;		// Alpha mask

int disp_font;		// Current font size: 8 or 16 (pixels-high)
int *colour;		// Array of colours mapped to current display: see colour_data

// Display areas
int d_tim_xx, d_tim_yy;	// Current-time display area
int d_tim_sx, d_tim_sy;
int d_mag_xx, d_mag_yy;	// Magnitude display area
int d_mag_sx, d_mag_sy;
int d_hsc_xx, d_hsc_yy;	// Horizontal scale display area
int d_hsc_sx, d_hsc_sy;
int d_vsc_xx, d_vsc_yy;	// Vertical scale display area
int d_vsc_sx, d_vsc_sy;

// Files and settings
Settings *file, *curr;	// array of loaded files, current file
int n_file;		// number of files loaded
int d_mode= -1;		// current display mode: -2 to 2

// Loaded analysis data
float *dat;		// currently loaded analysis data
Settings *dat_file;	// file it applies to 
int dat_off;		// file offset it applies to
FILE *dat_in;		// open file, or 0

// Current analysis
Settings *an_file;	// File this is analysis of
int an_off;		// Offset		
double an_midx, an_midy; // Position
double an_scale;        // Scale
int an_sx, an_sy;	// Pixel size
float *analysis;	// Analysis results

// Main loop
char *progdir;          // Directory in which binary/EXE is located
int rearrange;		// Set to request a complete rearrange/recalc/redraw of the screen
int redraw;		// Set to request a redraw of the screen


//
//      Utility functions
//

void 
error(char *fmt, ...) {
   va_list ap;
   va_start(ap, fmt);
   fprintf(stderr, PROGRAM ": ");
   vfprintf(stderr, fmt, ap);
   fprintf(stderr, "\n");
   exit(1);
}

void
errorSDL(char *fmt, ...) {
   va_list ap;
   va_start(ap, fmt);
   fprintf(stderr, PROGRAM ": ");
   vfprintf(stderr, fmt, ap);
   fprintf(stderr, "\n  %s\n", SDL_GetError());
   exit(1);
}

#define NL "\n"

void
usage() {
   error("Binaural beat visual analysis tool, version " VERSION
	 NL "Copyright (c) 2002-2005 Jim Peters, http://uazu.net, all rights reserved,"
	 NL "  released under the GNU GPL v2; see file COPYING"
	 NL
	 NL "Usage: bavsa-view [options]"
	 NL
	 NL "Options:"
	 NL "  -F <mode>     Run full-screen with the given mode, <wid>x<hgt>x<bpp>"
	 NL "                <bpp> may be 16 or 32.  For example: 800x600x16"
	 NL "  -W <size>     Run as a window with the given size: <wid>x<hgt>"
	 );
}

void 
warn(char *fmt, ...) {
   va_list ap;
   va_start(ap, fmt);
   vfprintf(stderr, fmt, ap);
   fprintf(stderr, "\n");
}

void *
Alloc(int size) {
   void *vp= calloc(1, size);
   if (!vp) error("Out of memory");
   return vp;
}

void *
StrDup(char *str) {
   char *cp= strdup(str);
   if (!cp) error("Out of memory");
   return cp;
}

char *
StrDupF(char *fmt, ...) {
   va_list ap;
   char buf[4096], *rv;
   int len;
   va_start(ap, fmt);
   len= vsnprintf(buf, sizeof(buf), fmt, ap);
   if (len < 0 || len >= sizeof(buf)-1)
      error("StrDupF exceeded buffer");
   rv= strdup(buf);
   if (!rv) error("Out of memory");
   return rv;
}



//
//	Main routine
//

int 
main(int ac, char **av) {
   char *p;
   int sx= 800, sy= 600, bpp= 0;	// Default is 800x600 resizable window
   SDL_Event ev;

   // Find which directory we're running in
   progdir= StrDup(av[0]);
   p= strchr(progdir, 0);
   while (p > progdir && p[-1] != '/' && p[-1] != '\\') *--p= 0;

   // Process arguments
   ac--; av++;
   while (ac > 0 && av[0][0] == '-' && av[0][1]) {
      char ch, dmy, *p= *av++ + 1; 
      ac--;
      while ((ch= *p++)) switch (ch) {
       case 'F':
	  if (ac-- < 1) usage();
	  if (3 != sscanf(*av++, "%dx%dx%d %c", &sx, &sy, &bpp, &dmy) ||
	      (bpp != 16 && bpp != 32))
	     error("Bad mode-spec: %s", av[-1]);
	  break;
       case 'W':
	  if (ac-- < 1) usage();
	  if (2 != sscanf(*av++, "%dx%d %c", &sx, &sy, &dmy))
	     error("Bad window size: %s", av[-1]);
	  break;
       default:	
	  usage();
      }
   }

   warn( PROGRAM ": Binaural beat visual analysis tool, version " VERSION);
   warn("  Copyright (c) 2002-2005 Jim Peters; released under the GNU GPL ");
   warn("  license v2.  See: http://uazu.net/bavsa/");
   warn("");

   // Load up list of files to view
   {
      DIR *dp;
      struct dirent *de;

      warn("Scanning directory: %s", progdir);
      dp= opendir(progdir);
      if (!dp) error("Error accessing directory: %s", progdir);

      n_file= 0; file= 0;

      while (0 != (de= readdir(dp))) {
         char *nam= de->d_name;
         p= strchr(nam, 0);
         if (p-4 > nam && 0 == strcasecmp(p-4, ".bav")) {
	    FILE *tmp;
	    curr= file;
	    file= Alloc((n_file+1) * sizeof(*file));
	    memcpy(file, curr, n_file * sizeof(*file));
	    curr= file + n_file++;
	    curr->fnam= StrDup(nam);
	    curr->path= StrDupF("%s%s", progdir, nam);
	    if (!(tmp= fopen(curr->path, "rb")))
	       error("Error opening file:\n  %s", curr->path);
	    if (1 != fread(&curr->h_magic, (char*)&curr->h_end-(char*)&curr->h_magic, 1, tmp))
	       error("Problems reading file header:\n  %s", curr->path);
	    if (0 != memcmp(curr->h_magic, "bavsa analysis\r\n", 16))
	       error("Bad magic on file:\n  %s", curr->path);
	    fclose(tmp);
	    curr->scale= 0;	// Force auto-adjust
	    curr->bri= 64.0;
	 }
      }

      closedir(dp);

      if (n_file == 0) 
	 error("There are no .bav files ready to display in this directory!\n"
	       "You should drag the WAV files in, then run bavsa to do the analysis first.\n");
      curr= file;
   }

   // Sort the files into name order; simple bubble sort
   {
      int a, b;
      for (a= 0; a<n_file; a++) {
	 for (b= a+1; b<n_file; b++) {
	    if (strcmp(file[a].fnam, file[b].fnam) > 0) {
	       Settings tmp;
	       memcpy(&tmp, &file[a], sizeof(tmp));
	       memcpy(&file[a], &file[b], sizeof(tmp));
	       memcpy(&file[b], &tmp, sizeof(tmp));
	    }
	 }
	 warn("  %s", file[a].fnam);
      }
   }

   // Initialize SDL
   if (0 > SDL_Init(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE))
      errorSDL("Couldn't initialize SDL");
   atexit(SDL_Quit);
   SDL_EnableUNICODE(1);		// So we get translated keypresses

   // Set key-repeat (alternatively, use SDL_DEFAULT_REPEAT_INTERVAL (==30))
   if (SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, 40))
      errorSDL("Couldn't enable key repeat");

   // Initialise graphics
   graphics_init(sx, sy, bpp);
   disp_font= 16;
   arrange_display();

   rearrange= 0;
   redraw= 1;

   // Main loop
   status("+\x8A Copyright (c) 2002-2005 Jim Peters. All rights reserved. "
	  "Released under the GNU GPL v2; see file \"COPYING\". ");

   while (1) {
      if (rearrange) {
	 disp_font= 16;
	 arrange_display();
	 clear_rect(0, 0, disp_sx, disp_sy, colour[0]);
	 update(0, 0, disp_sx, disp_sy);
	 draw_status();
	 rearrange= 0;
	 redraw= 1;
      }

      if (redraw) {
	 if (!curr->scale) auto_adjust();	// First time
	 //suspend_update= 1;
	 draw_tim();
	 draw_hsc();
	 draw_vsc();
	 draw_mag();
	 //suspend_update= 0;
	 //update(0, 0, disp_sx, disp_sy);
	 redraw= 0;
      }

      if (!SDL_WaitEvent(0)) 
	 errorSDL("Unexpected error waiting for events");
      
      // Process all outstanding events
      while (SDL_PollEvent(&ev)) {
	 int ch;
	 switch (ev.type) {
	  case SDL_KEYDOWN:
	     status("");	// Use keypress to clear temporary messages from status line
	     ch= ev.key.keysym.unicode;
	     
	     if (isdigit(ch)) {
		curr->bri= pow(2, 1.0 * ((ch=='0') ? 9 : ch-'1'));
		status("Brightness: %g", curr->bri); 
		redraw= 1; break;
	     }
	     
	     switch ((ch >= 32 && ch <= 126) ? tolower(ch) : ev.key.keysym.sym) {
	      case '+':
	      case '=':
		 curr->bri *= pow(2, 0.125);
		 status("Brightness: %g", curr->bri); 
		 redraw= 1; break;
	      case '-':
	      case '_':
		 curr->bri *= pow(2, -0.125);
		 status("Brightness: %g", curr->bri); 
		 redraw= 1; break;
	      case 'q':
		 exit(0);
	      case 'z':
	      case 'x':
		 curr->scale *= pow(2, ch == 'z' ? -0.5 : 0.5);
		 redraw= 1; break;
	      case 'a':
		 auto_adjust();
		 redraw= 1; break;
	      case 'g':
	      case 'h':
	      case 'j':
	      case 'k':
	      case 'l':
		 d_mode= (strchr("ghjkl01234", ch))[5]-'2';
		 redraw= 1; break;
	      case '[':
	      case ']':
		 {
		    int a;
		    for (a= 0; a<n_file && curr != file+a; a++) ;
		    a= (a + (ch == '[' ? -1 : 1) + n_file) % n_file;
		    curr= file + a;
		    status("Files: prev \x8C%s\x80 curr \x8A%s\x80 next \x8C%s\x80", 
			   file[(a-1+n_file)%n_file].fnam, 
			   curr->fnam,
			   file[(a+1)%n_file].fnam);
		    redraw= 1;
		 }
		 break;
              case SDLK_ESCAPE:
		 exit(0);
	      case SDLK_BACKSPACE:
	      case SDLK_PAGEUP:
		 if (curr->off > 0) { 
		    curr->off--;
		    redraw= 1;
		 }
		 break;
	      case SDLK_SPACE:
	      case SDLK_PAGEDOWN:
		 curr->off++;
		 redraw= 1; break;
	      case SDLK_LEFT:
		 curr->midx -= curr->scale * 0.25 * d_mag_sx;
		 redraw= 1; break;
	      case SDLK_RIGHT:
		 curr->midx += curr->scale * 0.25 * d_mag_sx;
		 redraw= 1; break;
	      case SDLK_UP:
		 curr->midy -= curr->scale * 0.25 * d_mag_sy * YFACT;
		 redraw= 1; break;
	      case SDLK_DOWN:
		 curr->midy += curr->scale * 0.25 * d_mag_sy * YFACT;
		 redraw= 1; break;
	      default:
		 if (ch >= 32 && ch <= 126) 
		    status("\x8A BAD KEY \x80 "
			   "Use: space backsp pgup pgdn arrows z x a 0-9 "
			   "- = + [ ] g h j k l esc q");
		 break;
	     }
	     break;
	  case SDL_MOUSEMOTION:
	     if (ev.motion.x >= d_mag_xx &&
		 ev.motion.x - d_mag_xx < d_mag_sx &&
		 ev.motion.y >= d_mag_yy &&
		 ev.motion.y - d_mag_yy < d_mag_sy)
		show_mag_status(ev.motion.x - d_mag_xx, ev.motion.y - d_mag_yy);
	     break;
	  case SDL_MOUSEBUTTONDOWN:
	     if (ev.button.x >= d_mag_xx &&
		 ev.button.x - d_mag_xx < d_mag_sx &&
		 ev.button.y >= d_mag_yy &&
		 ev.button.y - d_mag_yy < d_mag_sy) {
		curr->midx += (ev.button.x - d_mag_xx - d_mag_sx*0.5) * curr->scale;
		curr->midy += (ev.button.y - d_mag_yy - d_mag_sy*0.5) * curr->scale * YFACT;
		switch (ev.button.button) {
		 case SDL_BUTTON_LEFT:
		    curr->scale *= pow(2, -0.5);
		    break;
		 case SDL_BUTTON_RIGHT:
		    curr->scale *= pow(2, 0.5);
		    break;
		}
		redraw= 1;
	     }	     
	     break;
	  case SDL_MOUSEBUTTONUP:
	     break;
	  case SDL_VIDEORESIZE:
	     sx= ev.resize.w;
	     sy= ev.resize.h;
	     graphics_init(sx, sy, 0);
	     rearrange= 1;
	     break;
	  case SDL_QUIT:
	     exit(0);
	 }
      }

      // Loop ...
   }

   return 0;
}

//
//	Adjust to fit all in display
//

void 
auto_adjust() {
   double targ= curr->h_cnt * curr->h_step;
   curr->scale= 1;
   curr->midx= curr->h_step * (curr->h_f0 + (curr->h_cnt-1)*0.5);
   curr->midy= 0;
   while (curr->scale * d_mag_sx > targ) 
      curr->scale *= pow(2, -0.5);
   while (curr->scale * d_mag_sx < targ) 
      curr->scale *= pow(2, 0.5);
}


//
//	Update the analysis array according to any change that have
//	taken place
//   

void 
do_analysis() {
   // Drop out if nothing has changed
   if (an_file == curr &&
       an_off == curr->off &&
       an_midx == curr->midx &&
       an_midy == curr->midy &&
       an_scale == curr->scale &&
       an_sx == d_mag_sx && 
       an_sy == d_mag_sy && 
       analysis) 
      return;

   if (an_sx != d_mag_sx ||
       an_sy != d_mag_sy) {
      free(analysis);
      analysis= 0;
   }

   if (!analysis) {
      an_sx= d_mag_sx;
      an_sy= d_mag_sy;
      analysis= ALLOC_ARR(an_sx*an_sy, float);
   }

   an_file= curr;
   an_off= curr->off;
   an_midx= curr->midx;
   an_midy= curr->midy;
   an_scale= curr->scale;

   {
      int xx, yy;
      double yorg= d_mag_sy*0.5;
      double xorg= d_mag_sx*0.5;
      double ysca= curr->scale * YFACT;
      double xsca= curr->scale;
      double ymid= curr->midy;
      double xmid= curr->midx;
      double hstep= curr->h_step;
      int hf0= curr->h_f0;
      int hcnt= curr->h_cnt;

      memset(analysis, 0, an_sx*an_sy*sizeof(float));

      for (yy= 0; yy<an_sy; yy++) {
	 double y0= (yy-yorg-0.5)*ysca + ymid;
	 double y1= (yy-yorg+0.5)*ysca + ymid;
	 if (y1 < -50 || y0 > 50) continue;

	 for (xx= 0; xx<an_sx; xx++) {
	    double x0= (xx-xorg-0.5)*xsca + xmid;
	    double x1= (xx-xorg+0.5)*xsca + xmid;
	    double offR0= x0 + y0*0.5;
	    double offR1= x1 + y1*0.5;
	    double offL0= x0 - y1*0.5;
	    double offL1= x1 - y0*0.5;
	    int oL0= -hf0 + (int)floor(offL0 / hstep);
	    int oL1= -hf0 + (int)floor(offL1 / hstep + 0.99);
	    int oR0= -hf0 + (int)floor(offR0 / hstep);
	    int oR1= -hf0 + (int)floor(offR1 / hstep + 0.99);
	    double maxL= 0, maxR= 0;
	    
	    if (oL0 < 0) oL0= 0;
	    if (oL1 >= hcnt) oL1= hcnt-1;
	    if (oR0 < 0) oR0= 0;
	    if (oR1 >= hcnt) oR1= hcnt-1;
	    
	    // We analyse a diamond-shape which actually extends
	    // outside the pixel boundaries, touching the corners
	    while (oL0 <= oL1) {
	       if (dat[oL0] > maxL) maxL= dat[oL0];
	       oL0++;
	    }
	    while (oR0 <= oR1) {
	       if (dat[oR0+hcnt] > maxR) maxR= dat[oR0+hcnt];
	       oR0++;
	    }
	    
	    analysis[xx+yy*an_sx]= (maxL < maxR) ? maxL : maxR;
	 }
      }
   }
}	    


//
//      Analyse the area around a single displayed point (which may
//      cover many analysis points).  Coordinates are relative to
//      top-left of display area.  'wid' is the size to analyse; 1.0
//      analyses a diamond of twice the area of a single pixel,
//      touching the pixel's corners.  Returns the maximum found, and
//      returns the carrier/beat frequencies of that maximum in
//      apmax_*.
//

double apmax_carr, apmax_beat;

double
analyse_point(int xx, int yy, double wid) {
   double yorg= d_mag_sy*0.5;
   double xorg= d_mag_sx*0.5;
   double ysca= curr->scale * YFACT;
   double xsca= curr->scale;
   double ymid= curr->midy;
   double xmid= curr->midx;
   double hstep= curr->h_step;
   int hf0= curr->h_f0;
   int hcnt= curr->h_cnt;

   double y0= (yy-yorg-0.5*wid)*ysca + ymid;
   double y1= (yy-yorg+0.5*wid)*ysca + ymid;
   double x0= (xx-xorg-0.5*wid)*xsca + xmid;
   double x1= (xx-xorg+0.5*wid)*xsca + xmid;
   double offR0= x0 + y0*0.5;
   double offR1= x1 + y1*0.5;
   double offL0= x0 - y1*0.5;
   double offL1= x1 - y0*0.5;
   int oL0= -hf0 + (int)floor(offL0 / hstep);
   int oL1= -hf0 + (int)floor(offL1 / hstep + 0.99);
   int oR0= -hf0 + (int)floor(offR0 / hstep);
   int oR1= -hf0 + (int)floor(offR1 / hstep + 0.99);
   double maxL= 0, maxR= 0;
   int oLmax= 0, oRmax= 0;
   
   if (oL0 < 0) oL0= 0;
   if (oL1 >= hcnt) oL1= hcnt-1;
   if (oR0 < 0) oR0= 0;
   if (oR1 >= hcnt) oR1= hcnt-1;

   //warn("Analyse region L:%d-%d, R:%d-%d", oL0, oL1, oR0, oR1);
   
   // We analyse a diamond-shape which actually extends
   // outside the pixel boundaries, touching the corners
   while (oL0 <= oL1) {
      if (dat[oL0] >= maxL) { maxL= dat[oL0]; oLmax= oL0; }
      oL0++;
   }
   while (oR0 <= oR1) {
      if (dat[oR0+hcnt] >= maxR) { maxR= dat[oR0+hcnt]; oRmax= oR0; }
      oR0++;
   }

   apmax_carr= ((oLmax + oRmax) * 0.5 + hf0) * hstep;
   apmax_beat= (oLmax - oRmax) * hstep;

   return maxL < maxR ? maxL : maxR;
}


//
//	Show details corresponding to the current mouse position, for example:
//
//	CURSOR: beat 200.2+5.4/4.2, time 1:30.4, file ".........."
//

void 
show_mag_status(int xx, int yy) {
   double max1= analyse_point(xx, yy, 1.0);
   double carr1= apmax_carr;
   double beat1= apmax_beat;
   double max2= analyse_point(xx, yy, 8.0);
   double carr2= apmax_carr;
   double beat2= apmax_beat;
   char buf[1024], *p= buf;
   int tt= (int)floor(curr->off * curr->h_tstep + curr->h_win/2);

   p += sprintf(p, "\x8C" "CURSOR:\x80");

   p += sprintf(p, " time \x8C%d:%02d\x80", tt/60, tt%60);

   p += sprintf(p, " nearby peak \x8C%.2f%+.2f/%.2f\x80", 
		carr2, beat2, 100.0*max2);

   p += sprintf(p, " point \x8C%.2f%+.2f/%.2f\x80", 
		carr1, beat1, 100.0*max1);

   p += sprintf(p, " file \x8A%s\x80", curr->fnam);

   status(buf);
}

#endif

// END //
