//
//	Bavsa -- Binaural beat visual analysis tool
//
//        Copyright (c) 2004-2005 Jim Peters <http://uazu.net/>.
//        Released under the GNU GPL version 2 as published by the
//        Free Software Foundation.  See the file COPYING for details,
//        or visit <http://www.gnu.org/copyleft/gpl.html>.
//

#define VERSION "1.0.1"
#define PROGRAM "bavsa"
   
#define NL "\n"

char *usage_text= 
(PROGRAM " version " VERSION 
 NL "Usage: " PROGRAM " [-f <start>-<end>/<step>] [-t <window-dur-secs>/<step>]"
 NL "             [-r <sampling-rate>] [<files...>]"
 NL
 NL "Default is '-f 0-1500/0.1 -t 7/5 -r 44100', i.e. frequency range 0 to 1500Hz, "
 NL "  in steps of 0.1Hz, analysis window of duration 7 seconds, stepping forwards"
 NL "  5 seconds at a time, and expecting a WAV file sampled at 44100Hz."
 NL
 NL "If no files are listed, all WAV files in the *program* directory are"
 NL "analysed; this is for the benefit of Windows users."
 );

//
//	Platform-specifics
//

#ifndef T_LINUX
#ifndef T_MINGW
#error Please define T_LINUX or T_MINGW
#endif
#endif

#ifdef T_MINGW
#define EXIT_KEY
#endif

//
//	Saved analysis format:
//
//	  [16]C  "bavsa analysis\r\n"
//	  F64	 freq step (Hz)
//	  I32	 frequency range start (/step)
//	  I32	 frequency range width (/step)
//	  F64	 window-length (secs)
//	  F64	 time step (secs)
//	  {
//	    I32    magic 0x12345678
//	    F32 x (frequency range width)    left magnitudes
//	    F32 x (frequency range width)    right magnitudes
//	  } x (duration of file / time step)
//

#include <malloc.h>
#include <time.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dirent.h>
#include <complex.h>
#include <fftw3.h>

#define TWOPI 6.2831853071795864769252867665590057683943388

//
//	Support code
//

#define ALLOC(type) ((type*)Alloc(sizeof(type)))
#define ALLOC_ARR(cnt, type) ((type*)Alloc((cnt) * sizeof(type)))

void 
exit_key() {
   int ch;
   fprintf(stderr, "\nPress <RETURN> to continue: ");
   fflush(stderr);
   while ((ch= getchar()) != '\r' && ch != '\n') ;
}

void
error(char *fmt, ...) {
   va_list ap;
   va_start(ap, fmt);
   vfprintf(stderr, fmt, ap);
   fprintf(stderr, "\n");
#ifdef EXIT_KEY
   exit_key();
#endif
   exit(1);
}

void
warn(char *fmt, ...) {
   va_list ap;
   va_start(ap, fmt);
   vfprintf(stdout, fmt, ap);
   fprintf(stdout, "\n");
   fflush(stdout);
}

void 
usage() {
   error("%s", usage_text);
}
   
void *
Alloc(int size) {
   void *vp= calloc(1, size);
   if (!vp) error("Out of memory");
   return vp;
}

void *
StrDup(char *str) {
   char *cp= strdup(str);
   if (!cp) error("Out of memory");
   return cp;
}

char *
StrDupF(char *fmt, ...) {
   va_list ap;
   char buf[4096], *rv;
   int len;
   va_start(ap, fmt);
   len= vsnprintf(buf, sizeof(buf), fmt, ap);
   if (len < 0 || len >= sizeof(buf)-1)
      error("StrDupF exceeded buffer");
   rv= strdup(buf);
   if (!rv) error("Out of memory");
   return rv;
}

//
//	Write binary formats
//

void 
wrD(FILE *out, double val) {
   if (1 != fwrite(&val, sizeof(val), 1, out))
      error("Write error on analysis file");
}
void 
wrF(FILE *out, double dval) {
   float val= dval;
   if (1 != fwrite(&val, sizeof(val), 1, out))
      error("Write error on analysis file");
}
void 
wrI(FILE *out, int val) {
   if (1 != fwrite(&val, sizeof(val), 1, out))
      error("Write error on analysis file");
}


//
//	Globals
//

double rate;		// Base sampling rate
double i_f0, i_f1;	// Input freq range
double i_step;		// Input freq step
double i_wid, i_tstep;	// Window width, time step

int win_len;		// Length of window (samples)
int step_len;		// Step length (samples)
double *window;

int fft_len;		// Length of FFT (samples)
int f0, f1;		// Frequency offset range to output

int fft_ilen;		// Input length of FFT (samples)
int fft_olen;		// Output length of FFT (bins)
int fft_cnt;
double *fft_in;
double *fft_tmp;
fftw_complex *fft_out;
fftw_complex *fft_twid;
fftw_plan fft_plan;

int f_debug= 0;
char *progdir;		// Directory in which binary/EXE is located

//
//	Setup the 'pruned' FFT, which is quicker when we're not
//	interested in frequencies over e.g. 1500Hz in the output.
//

void
setup_fft(int ilen, int olen) {
   int cnt= ilen / olen;
   int i, j;
   fftw_r2r_kind kind= FFTW_R2HC;

   // Increase olen until we have a clean divisor into ilen
   while ((ilen/cnt)*cnt != ilen) cnt--;
   olen= ilen / cnt;

   fft_ilen= ilen;
   fft_olen= olen;
   fft_cnt= cnt;

   fft_in= fftw_malloc(fft_ilen * sizeof(double));
   fft_tmp= fftw_malloc(fft_ilen * sizeof(double));
   fft_out= fftw_malloc(fft_olen * sizeof(fftw_complex));
   fft_twid= fftw_malloc((olen-1)*(cnt-1) * sizeof(fftw_complex));

   if (!fft_in || !fft_out || !fft_twid) error("Out of memory");

   // This code based on http://www.fftw.org/pruned.html
   fft_plan= fftw_plan_many_r2r(1, &olen, cnt,
                                fft_in, NULL, cnt, 1,
                                fft_tmp, NULL, 1, olen,
                                &kind, FFTW_MEASURE);

   for (j= 1; j < cnt; ++j)
      for (i= 1; i < olen; ++i)
         fft_twid[(j-1)*(olen-1) + (i-1)]=
            cexp((I * FFTW_FORWARD * TWOPI/ilen) * (i*j));

   // Clear input array
   memset(fft_in, 0, sizeof(fft_in[0]) * fft_ilen);
}


//
//	FFT a chunk of data from fft_in[] to fft_out[]
//

void
do_fft() {
   fftw_execute(fft_plan);

   // This code based on http://www.fftw.org/pruned.html
   {
      int i, j;
      double *out1= fft_tmp;
      fftw_complex *out= fft_out;
      int K= fft_olen;
      int NoK= fft_cnt;

      out[0]= out1[0];
      for (i= 1; i+i < K; ++i) {
         double o1= out1[i], o2= out1[K-i];
         out[i]= o1 + I*o2;
         out[K-i]= o1 - I*o2;
      }
      if (i+i == K) /* Nyquist element */
         out[i]= out1[i];
      for (j= 1; j < NoK; ++j) {
         out[0] += out1[j*K];
         for (i= 1; i+i < K; ++i) {
            double o1= out1[i + j*K], o2= out1[K-i + j*K];
            out[i] += (o1 + I*o2) * fft_twid[(j-1)*(K-1) + (i-1)];
            out[K-i] += (o1 - I*o2) * fft_twid[(j-1)*(K-1) + (K-i-1)];
         }
         if (i+i == K) /* Nyquist element */
            out[i] += out1[i + j*K] * fft_twid[(j-1)*(K-1) + (i-1)];
      }
   }
}


//
//	Setup the analysis based on the 'i_*' vars and 'rate'
//

void 
setup_analysis() {
   int a;
   double gain;

   win_len= i_wid * rate;
   step_len= i_tstep * rate;
   window= ALLOC_ARR(win_len, double);
   
   // Blackman window, created slightly larger than space, taking into
   // account that first and last will be zero, and so can be dropped
   // off edge.
   for (a= 0; a<win_len; a++) {
      double th= TWOPI * ((a+1.0) / (win_len+1.0) - 0.5);
      window[a]= 0.42 + 0.5 * cos(th) + 0.08 * cos(th * 2.0);
   }

   // Select an fft_len with a step of i_step, i_step/2, i_step/3 etc,
   // to make it long enough for the window-length.
   for (a= 1; win_len > (fft_len= a*rate/i_step); a++) ;

   f0= (int)floor(i_f0/rate*fft_len);
   f1= (int)floor(0.9999 + i_f1/rate*fft_len);
   if (f1 > fft_len/2) f1= fft_len/2;
   if (f0 > f1) f0= f1;

   setup_fft(fft_len, f1+1);

   // Test gain of FFT and adjust window
   for (a= 0; a<win_len; a++)
      fft_in[a]= window[a] * 32767;
   do_fft();
   gain= cabs(fft_out[0]);
   for (a= 0; a<win_len; a++)
      window[a] /= gain;
}


//
//	Process chunk; analyse a chunk of 16-bit stereo data, and
//	output binary data to 'out'.
//

void 
do_analysis(short *sp, FILE *out) {
   int a, b;

   wrI(out, 0x12345678);
   for (b= 0; b<2; b++) {
      for (a= 0; a<win_len; a++) 
	 fft_in[a]= sp[a*2+b] * window[a];
      do_fft(fft_plan);
      for (a= f0; a<=f1; a++) 
	 wrF(out, cabs(fft_out[a]) * (a ? 2 : 1));
   }
}

	 
//
//	Wisdom handling
//

#define WISDOM_FNAM PROGRAM ".wis"

void 
read_wisdom() {
   char *fnam= StrDupF("%s%s", progdir, WISDOM_FNAM);
   FILE *fp= fopen(fnam, "r");
   if (fp) {
      fftw_import_wisdom_from_file(fp);
      fclose(fp);
   }
   free(fnam);
}

void 
write_wisdom() {
   char *fnam= StrDupF("%s%s", progdir, WISDOM_FNAM);
   FILE *fp= fopen(fnam, "w");
   if (!fp) error("Can't create wisdom file: %s", fnam);
   fftw_export_wisdom_to_file(fp);
   if (0 != fclose(fp)) 
      error("Can't create wisdom file: %s", fnam);
   free(fnam);
}

//
//	Skip over WAV headers, and check that any "fmt " section
//	represents 16-bit stereo 44100Hz audio.  Returns a strdup'd
//	error indication, or 0 for success.
//

char *
find_wav_data_start(FILE *in) {
   unsigned char buf[64];

   if (1 != fread(buf, 12, 1, in)) goto bad;
   if (0 != memcmp(buf, "RIFF", 4)) goto bad;
   if (0 != memcmp(buf+8, "WAVE", 4)) goto bad;

   while (1) {
      int len;
      int val;
      if (1 != fread(buf, 8, 1, in)) goto bad;
      if (0 == memcmp(buf, "data", 4)) return 0;          // We're in the right place!
      len= buf[4] + (buf[5]<<8) + (buf[6]<<16) + (buf[7]<<24);
      if (len & 1) len++;
      if (0 == memcmp(buf, "fmt ", 4)) {
         // Check the sample rate and number of channels
         if (1 != fread(buf, 16, 1, in)) goto bad;
         len -= 16;
	 val= buf[0] + (buf[1]<<8);
	 if (val != 1) return StrDupF("WAV file has unexpected format tag: %d", val);
	 val= buf[2] + (buf[3]<<8);
	 if (val != 2) return StrDupF("WAV file has %d channels, expecting 2", val);
         val= buf[4] + (buf[5]<<8) + (buf[6]<<16) + (buf[7]<<24);
	 if (val != rate) return StrDupF("WAV file sampled at %dHz, expecting %gHz"
					 " (see -r option)", val, rate);
	 val= buf[14] + (buf[15]<<8);
	 if (val != 16) return StrDupF("WAV file has %d-bit samples, expecting 16-bit", val);
      }
      if (len && 0 != fseek(in, len, SEEK_CUR)) goto bad;
   }

 bad:
   return StrDupF("WAV file has invalid headers");
}


//
//	Analyse a file XX.wav, writing the result to XX.bav.  If the
//	output file already exists and 'force' is not set, then
//	nothing is done.  'dnam' is a directory name to prepend, or 0.
//

void 
process(char *dnam, char *fnam, int force) {
   char *onam, *inam;
   char *p;
   FILE *in, *out;
   int scnt= 0;
   int ilen= win_len;
   int step= step_len;
   short *ibuf= ALLOC_ARR(ilen*2, short);
   short *iend= ibuf + ilen*2;
   short *ip= ibuf;
   char *err;
   int tim0, tim1, tdiff, tt;

   tim0= time(0);

   // Setup filenames
   inam= StrDupF("%s%s", dnam?dnam:"", fnam);
   onam= StrDupF("%s____", inam);
   p= strchr(onam, 0) - 4; *p= 0;
   if (p-4 > onam && 0 == strcasecmp(p-4, ".wav")) p -= 4;
   strcpy(p, ".bav");

   // See if output file already exists
   in= fopen(onam, "r");
   if (in) {
      fclose(in);
      if (!force) goto tidyup;
   }

   // Open and check input file
   in= fopen(fnam, "rb");
   if (!in) error("Can't open file: %s", fnam);
   err= find_wav_data_start(in);
   if (err) error("Problem reading WAV headers on '%s':\n  %s", fnam, err);

   // Open output file and write header
   out= fopen(onam, "wb");
   if (!out) error("Can't create file for writing: %s", onam);
   fprintf(out, "bavsa analysis\r\n");
   wrD(out, rate / fft_len);
   wrI(out, f0);
   wrI(out, f1-f0+1);
   wrD(out, win_len / rate);
   wrD(out, step_len / rate);

   // Loop until EOF
   while (1) {
      // Status
      tt= scnt/rate;
      printf("Processing %02d:%02d of '%s' ...\r", tt/60, tt%60, fnam);
      fflush(stdout);

      // Fill buffer
      while (ip < iend) {
	 ip += fread(ip, sizeof(short), iend-ip, in);
	 if (ip != iend) {
	    if (ferror(in)) error("Read error on input file: %s  ", fnam);
	    break;
	 }
      }
      if (ip != iend) break;	// Must be EOF

      // Analysis and output
      do_analysis(ibuf, out);

      // Move onto next segment
      if (step > ilen) {
	 fseek(in, (step-ilen) * 2 * sizeof(short), SEEK_CUR);
	 ip= ibuf;
      } else {
	 memmove(ibuf, ibuf + step*2, iend-(ibuf+step*2));
	 ip -= step*2;
      }
      scnt += step;
   }

   // Tidy up
   tim1= time(0);	
   tdiff= tim1-tim0;
   tt= scnt/rate;
   printf("Processed %dm%ds from '%s' in %dm%ds (%.1fx)\n",
	  tt/60, tt%60, fnam,
	  tdiff/60, tdiff%60, scnt*1.0/rate/tdiff);

   if (0 != fclose(out)) error("Write error on file: %s", onam);
   if (0 != fclose(in)) error("Input error on file: %s", fnam);
   free(ibuf);

 tidyup:
   free(inam);
   free(onam);
}


//
//	Main
//  

int 
main(int ac, char **av) {
   char *p;

   // Set defaults
   rate= 44100;
   i_f0= 0;
   i_f1= 1500;
   i_step= 0.1;
   i_wid= 7;
   i_tstep= 5;

   // Find which directory we're running in
   progdir= StrDup(av[0]);
   p= strchr(progdir, 0);
   while (p > progdir && p[-1] != '/' && p[-1] != '\\') *--p= 0;

   // Options
   ac--; av++;
   while (ac > 0 && av[0][0] == '-') {
      char ch, *p= 1+*av++; ac--;
      while ((ch= *p++)) switch (ch) {
       case 'd': f_debug++; break;
       case 'f': 
	  if (ac < 1 || 
	      3 != sscanf(av[0], "%lf-%lf/%lf %c", &i_f0, &i_f1, &i_step, &ch))
	     usage();
	  av++; ac--; break;
       case 't': 
	  if (ac < 1 || 
	      2 != sscanf(av[0], "%lf/%lf %c", &i_wid, &i_tstep, &ch))
	     usage();
	  av++; ac--; break;
       case 'r': 
	  if (ac < 1 || 
	      1 != sscanf(av[0], "%lf %c", &rate, &ch))
	     usage();
	  av++; ac--; break;
       default: 
	  usage();
      }
   }
   
   warn("\"" PROGRAM "\" binaural beat visual analysis tool, version " VERSION);
   warn("  Copyright (c) 2004-2005 Jim Peters; released under the GNU GPL ");
   warn("  license v2.  See: http://uazu.net/bavsa/");
   warn("");

   printf("Initialising FFTW engine ...\n"); 
   
   // Do all the FFT plan and table setup
   read_wisdom();
   setup_analysis();
   write_wisdom();

   // Process files
   if (ac) {
      while (ac-- > 0) 
	 process(0, *av++, 1);
   } else {
      DIR *dp;
      struct dirent *de;

      warn("Scanning directory: %s", progdir);
      dp= opendir(progdir);
      if (!dp) error("Error accessing directory");

      while ((de= readdir(dp))) {
	 char *nam= de->d_name;
	 p= strchr(nam, 0);
	 if (p-4 > nam && 0 == strcasecmp(p-4, ".wav"))
	    process(progdir, nam, 0);
      }

      closedir(dp);
   }

#ifdef EXIT_KEY
   exit_key();
#endif

   return 0;
}

// END //

